package main

import (
	"fmt"
	"github.com/google/gopacket"
	"github.com/google/gopacket/layers"
	"github.com/google/gopacket/pcap"
	"time"
)

// Filter for syn packets
const filter = "(tcp[tcpflags] & tcp-syn != 0)"

func handlePacket(handle *pcap.Handle, packet gopacket.Packet) {
	eth := packet.LinkLayer().(*layers.Ethernet)
	ipv4 := packet.NetworkLayer().(*layers.IPv4)
	tcp := packet.TransportLayer().(*layers.TCP)
	fmt.Printf("[%s] SYN packet received from %s on port %d. Replying with SYN+ACK...\n", time.Now().String(), ipv4.SrcIP, tcp.DstPort)

	// Build syn+ack response
	buf := gopacket.NewSerializeBuffer()
	err := gopacket.SerializeLayers(buf, gopacket.SerializeOptions{},
		&layers.Ethernet{
			DstMAC: eth.SrcMAC,
			SrcMAC: eth.DstMAC,
		},
		&layers.IPv4{
			SrcIP: ipv4.DstIP,
			DstIP: ipv4.SrcIP,
		},
		&layers.TCP{
			SrcPort: tcp.DstPort,
			DstPort: tcp.SrcPort,
			Seq:     tcp.Ack,
			Ack:     tcp.Seq + 1,
			SYN:     true,
			ACK:     true,
		})
	if err != nil {
		panic(err)
	}
	err = handle.WritePacketData(buf.Bytes())
	if err != nil {
		panic(err)
	}
}

func main() {
	handle, err := pcap.OpenLive("eth0", 1600, true, pcap.BlockForever) // Change eth0 for your device
	if err != nil {
		panic(err)
	}
	_ = handle.SetBPFFilter(filter)
	packetSource := gopacket.NewPacketSource(handle, handle.LinkType())
	for packet := range packetSource.Packets() {
		handlePacket(handle, packet)
	}
}
