  public static byte[] generateByteCode(List<Command> commands, Base64Converter converter) {
    final ByteArrayOutputStream payload = new ByteArrayOutputStream(128);

    // Initialize the printer: Clear buffers and reset states
    for (int i=0; i<40; i++) {
      payload.write(0x00);
      payload.write(Constants.ESC);
      payload.write('@');
    }

    for (final Command command : commands) {
      switch (command.getCommandId()) {
        case Align:
          payload.write(Constants.ESC);
          payload.write((byte) 'a');
          payload.write((byte) ((((Align) command).getType() == Align.Type.Left) ? 0 : ((((Align) command).getType() == Align.Type.Center) ? 1 : 2)));
          break;
        case Barcode:
          final Barcode barCode = (Barcode) command;
          // Set Bar Code Height first
          payload.write(Constants.GS);
          payload.write((byte) 0x68);
          payload.write((byte) barCode.getHeight());
          byte barcodeType = 0;
          // Print Bar Code
          // TODO: Add Error Handling for invalid data. For now we are just ignoring the invalid data
          switch (barCode.getType()) {
            case UPCA:
              barcodeType = 0;
              if (barCode.getCode().length() != 11) {
                System.out.println("PrinterJobHandler: Bar Code invalid length for " + barCode.getType().name() + ": " + barCode.getCode().length());
                continue;
              }
              break;
            case UPCE:
              barcodeType = 1;
              if (barCode.getCode().length() != 7) {
                System.out.println("PrinterJobHandler: Bar Code invalid length for " + barCode.getType().name() + ": " + barCode.getCode().length());
                continue;
              }
              break;
            case JAN13:
              barcodeType = 2;
              if (barCode.getCode().length() != 12) {
                System.out.println("PrinterJobHandler: Bar Code invalid length for " + barCode.getType().name() + ": " + barCode.getCode().length());
                continue;
              }
              break;
            case JAN8:
              barcodeType = 3;
              if (barCode.getCode().length() != 7) {
                System.out.println("PrinterJobHandler: Bar Code invalid length for " + barCode.getType().name() + ": " + barCode.getCode().length());
                continue;
              }
              break;
            case CODE39:
              barcodeType = 4;
              break;
            case ITF:
              barcodeType = 5;
              if (barCode.getCode().length() % 2 != 1) {
                System.out.println("PrinterJobHandler: Bar Code invalid length for " + barCode.getType().name() + ": " + barCode.getCode().length());
                continue;
              }
              break;
            case NW7:
              barcodeType = 6;
              break;
          }
          // Send start of Bar Code
          payload.write(Constants.GS);
          payload.write('k');
          payload.write(barcodeType);

          // Send Bar Code data
          for (Byte n : barCode.getCode().getBytes()) {
            payload.write(n);
          }

          payload.write((byte) 0x00);
          break;
        case Bitmap:
          if (converter != null) {
            Bitmap bmp = (Bitmap) command;
            BitSet imageData = RasterTools.imageDataToBits(converter.fromBase64(bmp.getRawb64()));
            final byte widthLSB = (byte) (bmp.getWidth() & 0xFF);
            final byte widthMSB = (byte) ((bmp.getWidth() >> 8) & 0xFF);

            // Set line height for 24 pt
            payload.write(Constants.ESC);
            payload.write('3');
            payload.write(24);

            int offset = 0;
            while (offset < bmp.getHeight()) {
              payload.write(Constants.ESC);
              payload.write('*');
              payload.write('!');
              payload.write(widthLSB);
              payload.write(widthMSB);

              int lineIndex = 0;
              byte[] line = new byte[3 * bmp.getWidth()];
              for (int x = 0; x < bmp.getWidth(); x++) {
                for (int k = 0; k < 3; k++) {
                  byte slice = 0;
                  for (int b = 0; b < 8; b++) {
                    final int y = (((offset / 8) + k) * 8) + b;
                    final int i = (y * bmp.getWidth()) + x;
                    boolean v = false;
                    if (i < imageData.size()) {
                      v = !imageData.get(i);
                    }
                    slice |= (byte) ((v ? 1 : 0) << (7 - b));
                  }
                  line[lineIndex + k] = slice;
                }
                lineIndex += 3;
              }

              for (byte b : line) {
                payload.write(b);
              }

              payload.write('\n');
              offset += 24;
            }

            // Set line height for default
            payload.write(Constants.ESC);
            payload.write('2');
          } else {
            System.out.println("PrinterJobHandler: Unable to print bitmap because there is no Base64Converter");
          }
          break;
        case Bold:
          payload.write(Constants.ESC);
          payload.write('E');
          payload.write(((Bold) command).getEnabled() ? (byte) 1 : (byte) 0);
          break;
        case Color:
          payload.write(Constants.ESC);
          payload.write('r');
          payload.write(((Color) command).getType() == Color.Type.Black ? (byte) 0 : (byte) 1);
          break;
        case Cut:
          payload.write('\n');
          payload.write('\n');
          payload.write('\n');
          payload.write('\n');
          payload.write('\n');
          payload.write(Constants.ESC);
          payload.write(((Cut) command).getType() == Cut.Type.Full ? ((byte) 'i') : ((byte) 'm'));
          break;
        case DashedLine:
          payload.write(Constants.ESC); // Align center
          payload.write((byte) 'a');
          payload.write((byte)  1);
          switch (((DashedLine) command).getFont()) {
            case FontB:
              for (final Byte n : DashedLine.dashedLineB.getBytes()) {
                payload.write(n);
              }
              break;
            default: // Defaults to FontA
              for (final Byte n : DashedLine.dashedLineA.getBytes()) {
                payload.write(n);
              }
              break;
          }
          payload.write(Constants.ESC); // Align Left
          payload.write((byte) 'a');
          payload.write((byte)  0);
          break;
        case Encoding:
          payload.write(Constants.FS);
          payload.write('(');
          payload.write('C');
          payload.write((byte) 0x02);
          payload.write((byte) 0x00);
          payload.write((byte) 0x30);
          payload.write(((Encoding) command).getType() == Encoding.Type.ASCII ? (byte) 1 : (byte) 2);
          break;
        case Font:
          payload.write(Constants.ESC);
          payload.write('M');
          switch (((Font) command).getFont()) {
            case FontA:
              payload.write((byte) 0);
              break;
            case FontB:
              payload.write((byte) 1);
              break;
            case FontC:
              payload.write((byte) 2);
              break;
            case FontD:
              payload.write((byte) 3);
              break;
            case FontE:
              payload.write((byte) 4);
              break;
            case SpecialFontA:
              payload.write((byte) 97);
              break;
            case SpecialFontB:
              payload.write((byte) 98);
              break;
          }
          break;
        case LineHeight:
          LineHeight cmd = (LineHeight) command;
          if (cmd.getLineHeight() == LineHeight.DEFAULT_SPACING) {
            payload.write(Constants.ESC);
            payload.write('2');
          } else {
            payload.write(Constants.ESC);
            payload.write('3');
            payload.write(cmd.getLineHeight());
          }
          break;
        case ReverseColor:
          payload.write(Constants.GS);
          payload.write('B');
          payload.write(((ReverseColor) command).getEnabled() ? (byte) 1 : (byte) 0);
          break;
        case Reset:
          payload.write(Constants.ESC);
          payload.write('@');
          break;
        case Margin:
          final Margin mrgn = (Margin) command;
          switch (mrgn.getType()) {
            case Left:
              payload.write(Constants.ESC);
              payload.write(Constants.SP);
              payload.write((byte) (mrgn.getSize() & 0xFF));
              break;
            case Right:
              payload.write(Constants.GS);
              payload.write((byte) 'L');
              payload.write((byte) ((mrgn.getSize() & 0xFFFF) % 0xFF));
              payload.write((byte) ((mrgn.getSize() & 0xFFFF) / 0xFF));
          }
          break;
        case SetTabSpace:
          payload.write(Constants.ESC);
          payload.write('D');
          final int tabSize = ((SetTabSpace) command).getTabSpace();
          for (int i = 1; i < 32; i++) {
            payload.write((byte) (i * tabSize));
          }
          payload.write(0x00);
          break;
        case SetTabPosition:
          payload.write(Constants.ESC);
          payload.write('D');

          final List<Integer> pos = ((SetTabPosition) command).getTabSpace();

          for (Integer po : pos) {
            payload.write(po.byteValue());
          }
          payload.write(0x00);

          break;
        case Underline:
          payload.write(Constants.ESC);
          payload.write('-');
          payload.write(((Underline) command).getEnabled() ? (byte) 1 : (byte) 0);
          break;
        case NewLine:
          payload.write((byte) 0x0A);
          break;
        case Text:
          final String text = ((Text) command).getText();
          for (final Byte n : text.getBytes()) {
            payload.write(n);
          }
          break;
      }
    }

    return payload.toByteArray();
  }