package main

import (
    "bufio"
    "fmt"
    "log"
    "net"
    "strings"
    "time"
)

const server = "euro.aprs2.net"
const port = 14580

const softwareName = "goaprs"
const softwareVersion = "0.1"

const callsign = "PU2NVX"
const passcode = "123456"
const notifyMessage = "HUEBR"

const SendInterval = time.Second * 60 * 5

var myCoordinate = []float32{-23.640000,-46.650000}

var lastCoordinateSent time.Time

var loggedIn = false

func Degrees2DMS(val float32) (deg int, min float32, n string) {
    n = "+"
    if val < 0 {
        val = -val
        n = "-"
    }

    deg = int(val)
    val -= float32(deg)
    val *= 60
    min = val

    return
}

func buildMessage(callsign, passcode, cmd string) string {
    return fmt.Sprintf("user %s pass %s vers %s %s %s", callsign, passcode, softwareName, softwareVersion, cmd)
}

func parseAPRS(msg string, conn net.Conn) {
    log.Printf("<- APRS: %s\n", msg)
}

func parseServer(msg string, conn net.Conn) {
    log.Printf("<- SERVER: %s\n", msg)
    if strings.Index(msg, "verified") > -1 {
        log.Printf("-- Logged in as %s\n", callsign)

        sendPosition(conn)
        lastCoordinateSent = time.Now()

        loggedIn = true
    }
}

func parseMessage(msg string, conn net.Conn) {
    if string(msg[0]) == "#" {
        // Server Message
        parseServer(msg[2:], conn)
        return
    }

    parseAPRS(msg, conn)
}

func formatPosition() string {
    lat := myCoordinate[0]
    lon := myCoordinate[1]

    latDeg, latMin, latV := Degrees2DMS(lat)
    lonDeg, lonMin, lonV := Degrees2DMS(lon)

    if latV == "-" {
        latV = "S"
    } else {
        latV = "N"
    }
    if lonV == "-" {
        lonV = "W"
    } else {
        lonV = "E"
    }

    return fmt.Sprintf("%02d%02.2f%s/%03d%02.2f%s-000/000/A=000000", latDeg, latMin, latV, lonDeg, lonMin, lonV)
}

func sendPosition(conn net.Conn) {
    log.Println("-- Sending Position")
    sendMessage(conn, buildMessage(callsign, passcode, "")) // First Line
    sendMessage(conn, fmt.Sprintf("%s>APDR15,TCPIP*:=%s %s", callsign, formatPosition(), notifyMessage))
}

func sendMessage(conn net.Conn, msg string) {
    log.Printf("-> %s\n" ,msg)
    _, err := conn.Write([]byte(msg + "\n"))
    if err != nil {
        panic(err)
    }
}

func main() {
    log.Printf("-- Connecting to %s:%d\n", server, port)
    conn, err := net.Dial("tcp", fmt.Sprintf("%s:%d", server, port))

    if err != nil {
        panic(err)
    }

    reader := bufio.NewReader(conn)
    time.Sleep(1 * time.Second)
    line, _, err := reader.ReadLine()

    if err != nil {
        panic(err)
    }
    log.Printf("<- %s\n", string(line))

    log.Println("-- Connected. Sending Login")

    msg := buildMessage(callsign, passcode, fmt.Sprintf("filter r/%f/%f/%d", myCoordinate[0], myCoordinate[1], 50))
    sendMessage(conn, msg)

    if err != nil {
        panic(err)
    }

    log.Println("-- Receiving messages")
    for {
        line, _, err := reader.ReadLine()
        if err != nil {
            panic(err)
        }
        parseMessage(string(line), conn)

        if time.Since(lastCoordinateSent) > SendInterval && loggedIn {
            sendPosition(conn)
            lastCoordinateSent = time.Now()
        }
    }
}